<?php
/**
 * Count Warning Handler for PHP 7.2+ Compatibility
 * Proyecto Survision - Filtro de warnings count() deprecated/changed
 * 
 * En PHP 7.2+, count() genera warnings cuando recibe parámetros no-countables
 * Este handler filtra específicamente estos warnings sin afectar otros errores
 * 
 * @author  Claude Analysis System
 * @version 1.0
 * @date    2025-01-23
 */

/**
 * Error handler personalizado para filtrar warnings específicos de count()
 * 
 * Este handler intercepta solo warnings de count() y los suprime,
 * dejando pasar todos los otros tipos de errores normalmente.
 * 
 * Casos que maneja:
 * - "count(): Parameter must be an array or an object that implements Countable"
 * - Variants del mismo mensaje en diferentes contextos
 * 
 * @param int    $errno   Nivel del error
 * @param string $errstr  Mensaje de error  
 * @param string $errfile Archivo donde ocurrió el error
 * @param int    $errline Línea donde ocurrió el error
 * @return bool           true = suprimir error, false = procesar normalmente
 */
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    
    // Solo procesar warnings (E_WARNING)
    if ($errno !== E_WARNING) {
        return false; // Dejar pasar otros tipos de error
    }
    
    // Patrones de mensajes de count() a filtrar
    $count_warning_patterns = array(
        'count(): Parameter must be an array',
        'count(): Parameter must be an object that implements Countable',
        'sizeof(): Parameter must be an array',
        'sizeof(): Parameter must be an object that implements Countable'
    );
    
    // Verificar si el warning es de count()
    foreach ($count_warning_patterns as $pattern) {
        if (strpos($errstr, $pattern) !== false) {
            // Log interno para debugging (opcional)
            error_log("Count Warning Suppressed: $errstr in $errfile:$errline");
            return true; // Suprimir este warning
        }
    }
    
    // No es un warning de count(), procesar normalmente
    return false;
    
}, E_WARNING); // Solo manejar warnings

/**
 * Error handler adicional para casos específicos de remoteFunctions
 * 
 * El sistema Remote Functions puede generar warnings en contextos específicos
 * que necesitan manejo especial debido al uso de eval()
 */
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    
    // Solo procesar warnings
    if ($errno !== E_WARNING) {
        return false;
    }
    
    // Verificar si viene de remoteFunctions.php con eval()
    if (strpos($errfile, 'remoteFunctions.php') !== false && 
        strpos($errfile, 'eval()') !== false &&
        strpos($errstr, 'count():') !== false) {
        
        // Log específico para remote functions
        error_log("Remote Functions Count Warning Suppressed: $errstr");
        return true; // Suprimir warning de remote functions
    }
    
    return false;
    
}, E_WARNING);

// Log de inicialización
if (PHP_VERSION_ID >= 70200) {
    error_log("Count Warning Handler cargado correctamente para PHP " . PHP_VERSION);
}

?>