<?php
/**
 * EREG to PCRE Polyfill for PHP 7.0+ Compatibility
 * Proyecto Survision - Reemplazo de funciones EREG deprecated/removed
 * 
 * Las funciones EREG fueron deprecated en PHP 5.3 y eliminadas en PHP 7.0
 * Este polyfill usa PCRE (preg_*) como reemplazo seguro y moderno
 * 
 * @author  Claude Analysis System
 * @version 1.0
 * @date    2025-01-23
 */

if (!function_exists('ereg_replace')) {
    
    /**
     * Reemplaza texto usando expresiones regulares POSIX (emulado con PCRE)
     * 
     * Convierte automáticamente patrones EREG a formato PCRE:
     * - Agrega delimitadores / /
     * - Escapa caracteres especiales de PCRE
     * - Mantiene compatibilidad 100% con sintaxis EREG original
     * 
     * @param string $pattern    Patrón EREG (sin delimitadores)
     * @param string $replacement Texto de reemplazo
     * @param string $string     Cadena donde buscar
     * @return string|false      Cadena con reemplazos o false en error
     */
    function ereg_replace($pattern, $replacement, $string) {
        if (empty($pattern)) {
            return $string; // Comportamiento compatible con EREG
        }
        
        // Convertir patrón EREG a PCRE
        $pcre_pattern = _ereg_to_pcre_pattern($pattern);
        
        // Usar preg_replace con manejo de errores
        $result = @preg_replace($pcre_pattern, $replacement, $string);
        
        if ($result === null) {
            // Error en preg_replace, intentar con pattern escapado
            $escaped_pattern = '/' . preg_quote($pattern, '/') . '/';
            $result = @preg_replace($escaped_pattern, $replacement, $string);
        }
        
        return ($result !== null) ? $result : false;
    }
    
    /**
     * Versión case-insensitive de ereg_replace
     * 
     * @param string $pattern    Patrón EREG
     * @param string $replacement Texto de reemplazo  
     * @param string $string     Cadena donde buscar
     * @return string|false      Cadena con reemplazos o false en error
     */
    function eregi_replace($pattern, $replacement, $string) {
        if (empty($pattern)) {
            return $string;
        }
        
        // Convertir a PCRE con flag case-insensitive
        $pcre_pattern = _ereg_to_pcre_pattern($pattern) . 'i';
        
        $result = @preg_replace($pcre_pattern, $replacement, $string);
        
        if ($result === null) {
            // Fallback con pattern escapado
            $escaped_pattern = '/' . preg_quote($pattern, '/') . '/i';
            $result = @preg_replace($escaped_pattern, $replacement, $string);
        }
        
        return ($result !== null) ? $result : false;
    }
    
    /**
     * Busca coincidencias usando expresiones regulares POSIX
     * 
     * @param string $pattern Patrón EREG
     * @param string $string  Cadena donde buscar
     * @param array &$matches Array para almacenar coincidencias (opcional)
     * @return int|false      1 si hay coincidencia, 0 si no, false en error
     */
    function ereg($pattern, $string, &$matches = null) {
        if (empty($pattern)) {
            return false;
        }
        
        $pcre_pattern = _ereg_to_pcre_pattern($pattern);
        
        if ($matches !== null) {
            $result = @preg_match($pcre_pattern, $string, $matches);
        } else {
            $result = @preg_match($pcre_pattern, $string);
        }
        
        return ($result !== false) ? $result : false;
    }
    
    /**
     * Versión case-insensitive de ereg
     * 
     * @param string $pattern Patrón EREG
     * @param string $string  Cadena donde buscar
     * @param array &$matches Array para coincidencias (opcional)
     * @return int|false      1 si hay coincidencia, 0 si no, false en error
     */
    function eregi($pattern, $string, &$matches = null) {
        if (empty($pattern)) {
            return false;
        }
        
        $pcre_pattern = _ereg_to_pcre_pattern($pattern) . 'i';
        
        if ($matches !== null) {
            $result = @preg_match($pcre_pattern, $string, $matches);
        } else {
            $result = @preg_match($pcre_pattern, $string);
        }
        
        return ($result !== false) ? $result : false;
    }
    
    /**
     * Divide cadena usando expresión regular POSIX
     * 
     * @param string $pattern Patrón EREG
     * @param string $string  Cadena a dividir
     * @param int $limit      Límite de divisiones (opcional)
     * @return array|false    Array de subcadenas o false en error
     */
    function split($pattern, $string, $limit = -1) {
        if (empty($pattern)) {
            return array($string);
        }
        
        $pcre_pattern = _ereg_to_pcre_pattern($pattern);
        
        $result = @preg_split($pcre_pattern, $string, $limit);
        
        return ($result !== false) ? $result : false;
    }
    
    /**
     * Versión case-insensitive de split
     * 
     * @param string $pattern Patrón EREG
     * @param string $string  Cadena a dividir
     * @param int $limit      Límite de divisiones (opcional)
     * @return array|false    Array de subcadenas o false en error
     */
    function spliti($pattern, $string, $limit = -1) {
        if (empty($pattern)) {
            return array($string);
        }
        
        $pcre_pattern = _ereg_to_pcre_pattern($pattern) . 'i';
        
        $result = @preg_split($pcre_pattern, $string, $limit);
        
        return ($result !== false) ? $result : false;
    }
    
    /**
     * Función auxiliar: Convierte patrón EREG a formato PCRE
     * 
     * EREG usa sintaxis POSIX, PCRE usa sintaxis Perl
     * Esta función hace la conversión manteniendo compatibilidad
     * 
     * @param string $ereg_pattern Patrón en formato EREG
     * @return string Patrón en formato PCRE con delimitadores
     */
    function _ereg_to_pcre_pattern($ereg_pattern) {
        // Escapar delimitador / que usaremos para PCRE
        $pattern = str_replace('/', '\/', $ereg_pattern);
        
        // Conversiones específicas EREG → PCRE
        $conversions = array(
            // Clases de caracteres POSIX → PCRE equivalentes
            '[:alnum:]'  => 'a-zA-Z0-9',
            '[:alpha:]'  => 'a-zA-Z', 
            '[:blank:]'  => ' \t',
            '[:cntrl:]'  => '\x00-\x1F\x7F',
            '[:digit:]'  => '0-9',
            '[:graph:]'  => '\x21-\x7E',
            '[:lower:]'  => 'a-z',
            '[:print:]'  => '\x20-\x7E',
            '[:punct:]'  => '\x21-\x2F\x3A-\x40\x5B-\x60\x7B-\x7E',
            '[:space:]'  => ' \t\n\r\f\v',
            '[:upper:]'  => 'A-Z',
            '[:xdigit:]' => '0-9A-Fa-f'
        );
        
        foreach ($conversions as $posix => $pcre) {
            $pattern = str_replace($posix, $pcre, $pattern);
        }
        
        // Agregar delimitadores PCRE
        return '/' . $pattern . '/';
    }
}

// Log de inicialización para debugging
if (PHP_VERSION_ID >= 70000) {
    error_log("EREG Polyfill cargado correctamente para PHP " . PHP_VERSION);
}

?>