<?php
/**
 * MySQL to MySQLi Polyfill for PHP 7.* Compatibility
 * Proyecto Survision - Migración PHP 5.4 → 7.*
 * 
 * Este polyfill proporciona compatibilidad 100% con funciones MySQL eliminadas
 * usando MySQLi por debajo. Mantiene la misma API y comportamiento.
 * 
 * @author  Claude Analysis System
 * @version 1.0
 * @date    2025-01-23
 */

// Variable global para mantener compatibilidad con código legacy
// que no pasa el parámetro de conexión
$mysql_link = null;

/**
 * Verificar que estamos en PHP 7.* donde mysql_* no existe
 * Solo definir funciones si no existen (evita conflictos)
 */
if (!function_exists('mysql_connect')) {
    
    /**
     * Establece conexión a servidor MySQL
     * 
     * Equivale a mysqli_connect() pero mantiene la API original
     * Guarda la conexión globalmente para funciones que no especifican $link
     * 
     * @param string $server   Servidor MySQL (ej: localhost)
     * @param string $username Usuario de base de datos  
     * @param string $password Contraseña
     * @param bool   $new_link Si crear nueva conexión (ignorado en MySQLi)
     * @param int    $client_flags Flags de cliente (ignorado)
     * @return mysqli|false Conexión MySQLi o false en error
     */
    function mysql_connect($server, $username, $password, $new_link = false, $client_flags = 0) {
        global $mysql_link;
        
        // Usar @ para suprimir warnings y manejar errores manualmente
        $link = @mysqli_connect($server, $username, $password);
        
        if (!$link) {
            // Mantener comportamiento original: retornar false en error
            return false;
        }
        
        // Guardar conexión global para funciones que no especifican $link
        // Esto es crucial para compatibilidad con código legacy
        if (!$mysql_link) {
            $mysql_link = $link;
        }
        
        return $link;
    }
    
    /**
     * Conexión persistente MySQL (legacy)
     * 
     * MySQLi no soporta conexiones persistentes como MySQL original
     * Fallback: usar conexión regular
     * 
     * @param string $server   Servidor MySQL
     * @param string $username Usuario  
     * @param string $password Contraseña
     * @param int    $client_flags Flags (ignorado)
     * @return mysqli|false Conexión o false
     */
    function mysql_pconnect($server, $username, $password, $client_flags = 0) {
        // MySQLi no tiene persistent connections como MySQL viejo
        // Usar conexión regular como fallback seguro
        return mysql_connect($server, $username, $password);
    }
    
    /**
     * Selecciona base de datos activa
     * 
     * Importante: MySQLi invierte el orden de parámetros vs MySQL original
     * MySQL: mysql_select_db($database, $link)  
     * MySQLi: mysqli_select_db($link, $database)
     * 
     * @param string $database Nombre de la base de datos
     * @param mysqli $link     Conexión (opcional, usa global si no se especifica)
     * @return bool True en éxito, false en error
     */
    function mysql_select_db($database, $link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        // Parámetros invertidos: MySQLi quiere ($link, $database)
        return @mysqli_select_db($connection, $database);
    }
    
    /**
     * Ejecuta consulta MySQL
     * 
     * Función más usada - 1000+ ocurrencias en el proyecto
     * MySQLi mantiene el mismo orden de parámetros que MySQL original
     * 
     * @param string $query Consulta SQL
     * @param mysqli $link  Conexión (opcional)
     * @return mysqli_result|bool Resultado de consulta o false
     */
    function mysql_query($query, $link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        // MySQLi mantiene el mismo orden: query, link
        return @mysqli_query($connection, $query);
    }
    
    /**
     * Obtiene fila como array asociativo y numérico
     * 
     * Comportamiento idéntico a MySQL original
     * Retorna: array(0=>"valor1", "campo1"=>"valor1", 1=>"valor2", "campo2"=>"valor2")
     * 
     * @param mysqli_result $result Resultado de consulta
     * @param int $result_type Tipo de array (MYSQL_BOTH por defecto)
     * @return array|false Array con datos o false si no hay más filas
     */
    function mysql_fetch_array($result, $result_type = MYSQL_BOTH) {
        if (!$result) {
            return false;
        }
        
        // Convertir constantes MySQL a MySQLi
        $type = MYSQLI_BOTH;  // Por defecto: índices numéricos y asociativos
        if ($result_type === MYSQL_ASSOC) {
            $type = MYSQLI_ASSOC;  // Solo asociativo
        } elseif ($result_type === MYSQL_NUM) {
            $type = MYSQLI_NUM;    // Solo numérico
        }
        
        return @mysqli_fetch_array($result, $type);
    }
    
    /**
     * Obtiene fila como array asociativo únicamente
     * 
     * Equivale a mysql_fetch_array($result, MYSQL_ASSOC)
     * Retorna: array("campo1"=>"valor1", "campo2"=>"valor2")
     * 
     * @param mysqli_result $result Resultado de consulta  
     * @return array|false Array asociativo o false
     */
    function mysql_fetch_assoc($result) {
        if (!$result) {
            return false;
        }
        return @mysqli_fetch_assoc($result);
    }
    
    /**
     * Obtiene fila como array numérico únicamente
     * 
     * Equivale a mysql_fetch_array($result, MYSQL_NUM)
     * Retorna: array(0=>"valor1", 1=>"valor2", 2=>"valor3")
     * 
     * @param mysqli_result $result Resultado de consulta
     * @return array|false Array numérico o false  
     */
    function mysql_fetch_row($result) {
        if (!$result) {
            return false;
        }
        return @mysqli_fetch_row($result);
    }
    
    /**
     * Cuenta filas en resultado
     * 
     * Importante: Solo funciona con SELECT, no con INSERT/UPDATE/DELETE
     * Para esos usar mysql_affected_rows()
     * 
     * @param mysqli_result $result Resultado de SELECT
     * @return int|false Número de filas o false en error
     */
    function mysql_num_rows($result) {
        if (!$result) {
            return false;
        }
        return @mysqli_num_rows($result);
    }
    
    /**
     * Cuenta columnas en resultado
     * 
     * Útil para procesar resultados dinámicamente sin conocer estructura
     * 
     * @param mysqli_result $result Resultado de consulta
     * @return int|false Número de columnas o false
     */
    function mysql_num_fields($result) {
        if (!$result) {
            return false;
        }
        return @mysqli_num_fields($result);
    }
    
    /**
     * Obtiene nombre de columna por índice
     * 
     * Usado en database.php:262 para obtener nombres de campos
     * MySQLi usa object property vs MySQL que era función directa
     * 
     * @param mysqli_result $result Resultado de consulta
     * @param int $field_offset Índice de columna (0-based)
     * @return string|false Nombre de campo o false
     */
    function mysql_field_name($result, $field_offset) {
        if (!$result) {
            return false;
        }
        
        // MySQLi usa objeto field info vs función directa de MySQL
        $field_info = @mysqli_fetch_field_direct($result, $field_offset);
        return $field_info ? $field_info->name : false;
    }
    
    /**
     * Escapa string para uso seguro en SQL
     * 
     * CRÍTICO: Previene SQL injection
     * Usado 1000+ veces en database.php
     * 
     * @param string $unescaped_string String a escapar
     * @param mysqli $link Conexión (opcional)
     * @return string String escapado y seguro
     */
    function mysql_real_escape_string($unescaped_string, $link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            // Fallback: addslashes si no hay conexión
            // No es tan seguro como mysqli_real_escape_string pero funciona
            return addslashes($unescaped_string);
        }
        
        return @mysqli_real_escape_string($connection, $unescaped_string);
    }
    
    /**
     * Obtiene ID del último INSERT con AUTO_INCREMENT
     * 
     * Esencial para database.php que lo usa en insertar() e insertar2()
     * Comportamiento idéntico a MySQL original
     * 
     * @param mysqli $link Conexión (opcional)
     * @return int|false Último ID insertado o false
     */
    function mysql_insert_id($link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        return @mysqli_insert_id($connection);
    }
    
    /**
     * Cuenta filas afectadas por último INSERT/UPDATE/DELETE
     * 
     * Diferencia vs mysql_num_rows():
     * - mysql_num_rows(): para SELECT (cuenta resultados)
     * - mysql_affected_rows(): para INSERT/UPDATE/DELETE (cuenta cambios)
     * 
     * @param mysqli $link Conexión (opcional)
     * @return int|false Filas afectadas o false
     */
    function mysql_affected_rows($link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        return @mysqli_affected_rows($connection);
    }
    
    /**
     * Obtiene último mensaje de error MySQL
     * 
     * Crítico para debugging y manejo de errores
     * database.php lo usa en múltiples die() statements
     * 
     * @param mysqli $link Conexión (opcional)
     * @return string Mensaje de error o string vacío
     */
    function mysql_error($link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return 'No MySQL connection established';
        }
        
        return @mysqli_error($connection);
    }
    
    /**
     * Obtiene código numérico del último error
     * 
     * Complementa mysql_error() para manejo programático de errores
     * 
     * @param mysqli $link Conexión (opcional)
     * @return int Código de error (0 = sin error)
     */
    function mysql_errno($link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return 0;
        }
        
        return @mysqli_errno($connection);
    }
    
    /**
     * Cierra conexión MySQL
     * 
     * Importante para cleanup y liberación de recursos
     * database.php lo usa en cerrar() y __destruct
     * 
     * @param mysqli $link Conexión (opcional)
     * @return bool True en éxito
     */
    function mysql_close($link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        $result = @mysqli_close($connection);
        
        // Limpiar variable global si esta era la conexión global
        if ($connection === $mysql_link) {
            $mysql_link = null;
        }
        
        return $result;
    }
    
    /**
     * Libera memoria de resultado de consulta
     * 
     * Importante para evitar memory leaks en aplicaciones de larga duración
     * database.php lo usa en liberar() y cerrar()
     * 
     * @param mysqli_result $result Resultado a liberar
     * @return bool True en éxito
     */
    function mysql_free_result($result) {
        if (!$result) {
            return false;
        }
        return @mysqli_free_result($result);
    }
    
    /**
     * Elimina base de datos completa
     * 
     * PELIGROSO: Solo para funciones administrativas
     * Usado en database.php:154 - función borrar()
     * 
     * @param string $database_name Nombre de BD a eliminar
     * @param mysqli $link Conexión (opcional)  
     * @return bool True si se eliminó correctamente
     */
    function mysql_drop_db($database_name, $link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        // Escapar nombre de BD para seguridad
        $safe_db_name = mysqli_real_escape_string($connection, $database_name);
        $query = "DROP DATABASE `$safe_db_name`";
        
        return @mysqli_query($connection, $query);
    }
    
    /**
     * Función legacy de escaping (menos segura)
     * 
     * DEPRECATED en MySQL original, pero tools.php:457 la usa
     * Fallback simple para compatibilidad
     * 
     * @param string $string String a escapar
     * @return string String escapado con addslashes
     */
    function mysql_escape_string($string) {
        // MySQL original: aplicaba addslashes básico
        // No tan seguro como mysql_real_escape_string pero compatible
        return addslashes($string);
    }
    
    // Definir constantes MySQL si no existen
    // Necesarias para mysql_fetch_array($result, MYSQL_ASSOC)
    if (!defined('MYSQL_ASSOC')) {
        define('MYSQL_ASSOC', MYSQLI_ASSOC);    // 1: Solo array asociativo
    }
    if (!defined('MYSQL_NUM')) {
        define('MYSQL_NUM', MYSQLI_NUM);        // 2: Solo array numérico  
    }
    if (!defined('MYSQL_BOTH')) {
        define('MYSQL_BOTH', MYSQLI_BOTH);      // 3: Ambos (por defecto)
    }
}

/**
 * Funciones de compatibilidad adicional
 * Para casos edge que podrían aparecer
 */

if (!function_exists('mysql_get_client_info')) {
    /**
     * Información del cliente MySQL
     * @return string Versión del cliente
     */
    function mysql_get_client_info() {
        return mysqli_get_client_info();
    }
}

if (!function_exists('mysql_get_server_info')) {
    /**
     * Información del servidor MySQL  
     * @param mysqli $link Conexión
     * @return string|false Versión del servidor
     */
    function mysql_get_server_info($link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        return $connection ? mysqli_get_server_info($connection) : false;
    }
    
    /**
     * Establece el charset de la conexión MySQL
     * 
     * CRÍTICO: Función requerida por database.php para establecer charset UTF-8
     * Sin esta función el login del CMS falla con Fatal Error
     * 
     * @param string $charset Charset a establecer (ej: 'utf8', 'utf8mb4')
     * @param mysqli $link Conexión (opcional)
     * @return bool True en éxito, false en error
     */
    function mysql_set_charset($charset, $link = null) {
        global $mysql_link;
        $connection = $link ? $link : $mysql_link;
        
        if (!$connection) {
            return false;
        }
        
        return @mysqli_set_charset($connection, $charset);
    }
}

// Log de inicialización para debugging
if (PHP_VERSION_ID >= 70000) {
    error_log("MySQL Polyfill cargado correctamente para PHP " . PHP_VERSION);
}

?>